<?php
/**
 * Simple License API Server (SQLite) - v1.0
 * Endpoints (POST):
 *  - action=validate   {license_key, site, plugin, version}
 *  - action=activate   {license_key, site, plugin, version}
 *  - action=deactivate {license_key, site}
 *  - action=generate   {admin_secret, plan, max_activations, expires_at(YYYY-MM-DD), notes(optional)}
 *  - action=list       {admin_secret}
 * Response: JSON -> {status: 'valid'|'invalid'|'expired'|'limit'|'ok'|'error', message?, data?}
 */

header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Dhaka');

require_once __DIR__ . '/config.php';

try {
    $pdo = new PDO('sqlite:' . __DIR__ . '/license.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    init_db($pdo);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status'=>'error','message'=>'DB error: '.$e->getMessage()]);
    exit;
}

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'validate':
        validate_license($pdo);
        break;
    case 'activate':
        activate_license($pdo);
        break;
    case 'deactivate':
        deactivate_license($pdo);
        break;
    case 'generate':
        admin_only();
        generate_license($pdo);
        break;
    case 'list':
        admin_only();
        list_licenses($pdo);
        break;
    default:
        echo json_encode(['status'=>'error','message'=>'Unknown action']);
}

function init_db(PDO $pdo){
    $pdo->exec("CREATE TABLE IF NOT EXISTS licenses (
        license_key TEXT PRIMARY KEY,
        plan TEXT NOT NULL,
        max_activations INTEGER NOT NULL DEFAULT 1,
        expires_at TEXT NULL,
        created_at TEXT NOT NULL,
        notes TEXT NULL
    )");
    $pdo->exec("CREATE TABLE IF NOT EXISTS activations (
        license_key TEXT NOT NULL,
        site TEXT NOT NULL,
        activated_at TEXT NOT NULL,
        UNIQUE(license_key, site)
    )");
}

function normalize_site($site){
    $site = trim(strtolower($site));
    $site = preg_replace('#^https?://#','',$site);
    $site = rtrim($site,'/');
    return $site;
}

function get_license(PDO $pdo, $key){
    $stmt = $pdo->prepare("SELECT * FROM licenses WHERE license_key = :k");
    $stmt->execute([':k'=>$key]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function count_activations(PDO $pdo, $key){
    $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM activations WHERE license_key = :k");
    $stmt->execute([':k'=>$key]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return intval($row ? $row['c'] : 0);
}

function is_activated_on(PDO $pdo, $key, $site){
    $stmt = $pdo->prepare("SELECT 1 FROM activations WHERE license_key = :k AND site = :s");
    $stmt->execute([':k'=>$key, ':s'=>$site]);
    return (bool)$stmt->fetchColumn();
}

function validate_core(PDO $pdo, $key, $site){
    $lic = get_license($pdo, $key);
    if (!$lic){
        return ['status'=>'invalid','message'=>'License not found'];
    }
    if (!empty($lic['expires_at']) && strtotime($lic['expires_at'].' 23:59:59') < time()){
        return ['status'=>'expired','message'=>'License expired'];
    }
    $activations = count_activations($pdo, $key);
    $max = intval($lic['max_activations']);
    if ($activations >= $max && !is_activated_on($pdo, $key, $site)){
        return ['status'=>'limit','message'=>'Activation limit reached'];
    }
    return ['status'=>'valid','message'=>'OK','license'=>$lic];
}

function validate_license(PDO $pdo){
    $key   = strtoupper(trim($_POST['license_key'] ?? ''));
    $site  = normalize_site($_POST['site'] ?? '');
    $plugin= trim($_POST['plugin'] ?? '');
    $ver   = trim($_POST['version'] ?? '');

    if (!$key || !$site){
        echo json_encode(['status'=>'error','message'=>'Missing license_key or site']); return;
    }
    $res = validate_core($pdo, $key, $site);
    echo json_encode(['status'=>$res['status'], 'message'=>$res['message']]);
}

function activate_license(PDO $pdo){
    $key   = strtoupper(trim($_POST['license_key'] ?? ''));
    $site  = normalize_site($_POST['site'] ?? '');
    if (!$key || !$site){
        echo json_encode(['status'=>'error','message'=>'Missing license_key or site']); return;
    }
    $check = validate_core($pdo, $key, $site);
    if ($check['status'] !== 'valid'){
        echo json_encode(['status'=>$check['status'],'message'=>$check['message']]); return;
    }
    try{
        $stmt = $pdo->prepare("INSERT OR IGNORE INTO activations(license_key, site, activated_at) VALUES(:k,:s,:t)");
        $stmt->execute([':k'=>$key, ':s'=>$site, ':t'=>date('Y-m-d H:i:s')]);
        echo json_encode(['status'=>'ok','message'=>'Activated']);
    } catch (Exception $e){
        echo json_encode(['status'=>'error','message'=>$e->getMessage()]);
    }
}

function deactivate_license(PDO $pdo){
    $key   = strtoupper(trim($_POST['license_key'] ?? ''));
    $site  = normalize_site($_POST['site'] ?? '');
    if (!$key || !$site){
        echo json_encode(['status'=>'error','message'=>'Missing license_key or site']); return;
    }
    $stmt = $pdo->prepare("DELETE FROM activations WHERE license_key = :k AND site = :s");
    $stmt->execute([':k'=>$key, ':s'=>$site]);
    echo json_encode(['status'=>'ok','message'=>'Deactivated']);
}

function random_key(){
    $alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $out='';
    for($i=0;$i<16;$i++){ $out .= $alphabet[random_int(0, strlen($alphabet)-1)]; }
    return substr($out,0,4).'-'.substr($out,4,4).'-'.substr($out,8,4).'-'.substr($out,12,4);
}

function admin_only(){
    $secret = $_POST['admin_secret'] ?? '';
    if ($secret !== ADMIN_SECRET){
        http_response_code(403);
        echo json_encode(['status'=>'error','message'=>'Forbidden']); exit;
    }
}

function generate_license(PDO $pdo){
    $plan   = trim($_POST['plan'] ?? 'pro');
    $max    = intval($_POST['max_activations'] ?? 1);
    $exp    = trim($_POST['expires_at'] ?? ''); // YYYY-MM-DD or empty
    $notes  = trim($_POST['notes'] ?? '');
    $key    = strtoupper($_POST['license_key'] ?? random_key());

    if (!preg_match('/^[A-Z0-9]{4}(-[A-Z0-9]{4}){3}$/', $key)){
        echo json_encode(['status'=>'error','message'=>'Invalid key format']); return;
    }
    try{
        $stmt = $pdo->prepare("INSERT INTO licenses(license_key, plan, max_activations, expires_at, created_at, notes) VALUES(:k,:p,:m,:e,:c,:n)");
        $stmt->execute([
            ':k'=>$key, ':p'=>$plan, ':m'=>$max,
            ':e'=>$exp ?: null, ':c'=>date('Y-m-d H:i:s'), ':n'=>$notes ?: null
        ]);
        echo json_encode(['status'=>'ok','message'=>'Generated','data'=>['license_key'=>$key]]);
    } catch (Exception $e){
        echo json_encode(['status'=>'error','message'=>$e->getMessage()]);
    }
}

function list_licenses(PDO $pdo){
    $stmt = $pdo->query("SELECT l.*, (SELECT COUNT(*) FROM activations a WHERE a.license_key=l.license_key) AS activations FROM licenses l ORDER BY created_at DESC");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    echo json_encode(['status'=>'ok','data'=>$rows]);
}
